/*
 * cPbcastLayer.cpp
 *
 * THE pbcast layer.
 */
#include "cPbcastLayer.h"
#include "cPbcastLayerParam.h"
#include "cPbcastView.h"
#include "cSortedMsgQueue.h"
#include "cSubnetTree.h"

#include "Endpoint/cEndpointFactory.h"
#include "Interfaces/cView.h"
#include "Util/cRegisterList.h"
#include "Util/gError.h"
#include "Protocol Stack/cProtocolStack.h"
#include "Protocol Stack/cBufferManager.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "Network Layer/cIPEndpoint.h"

#include "cPbcastInfo.h"
#include "cPbcastNode.h"

#define MAX_REGISTER		8
#define	MAX_MSG_GAP			64
#define MIN_GOSSIP_INTERVAL	500
#define DISCOVERY_INTERVAL	3000
#define	SUBNET_TREE_FANOUT	2
#define	MAX_FANOUT			8
#define SCHEDULE_INTERVAL	4000
#define T_FAIL				10000
#define T_CLEANUP			2*T_FAIL
#define T_MSG_CLEANUP		4*T_FAIL

/*
 * cPbcastLayer::Init()
 *
 * Purpose:	Initializes the pbcast layer.
 * IN:		layerBelow	-> The layer below this layer.
 *			param		-> The parameters intended for this layer.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer is ready to rock.
 * Return:	true if success, else false.
 */
bool cPbcastLayer::Init(cLayer* layerBelow, cParam* param)
{
	mStarted	  = false;
	mViewList     = NULL;
	mRegisterList = NULL;
	mMyNode		  = NULL;
	mMsgQueue	  = NULL;
	mSeqNum		  = 0;

	cPbcastLayerParam*	nParam = (cPbcastLayerParam *)param;
	_SetupParams(nParam);

	// Init our pointer to the protocol stack.
	mProtocolStack = param->mProtocolStack;

	// Set up the msg queue
	mMsgQueue = new cSortedMsgQueue(nParam->mMsgQueueSize, T_MSG_CLEANUP, nParam->mMsgQueueGrowInc);
	if(!mMsgQueue)
	{
		return false;
	}

	// Set up register list.
	mRegisterList = new cRegisterList(MAX_REGISTER);
	if(!mRegisterList)
	{
		return false;
	}

	// Set up the view list.
	mViewList = new cRegisterList(MAX_REGISTER);
	if(!mViewList)
	{
		return false;
	}

	// Set up the error callback list.
	mErrorCallbackList = new cRegisterList(MAX_REGISTER);
	if(!mErrorCallbackList)
	{
		return false;
	}

	// Set up sending group
	mSendGroup = new cGroup(16);
	if(!mSendGroup)
	{
		return false;
	}

	// Set up retransmit request group.
	mRetransReqGroup = new cGroup(1);
	if(!mRetransReqGroup)
	{
		return false;
	}
	_ResetRetransmitReqMsg();

	// Copy layer info from below
	mLayerBelow = layerBelow;
	if(layerBelow)
	{
		cLayerInfo* belowInfo;
		belowInfo = mLayerBelow->GetLayerInfo();
		mLayerInfo = *belowInfo;
	}
	mLayerInfo.SetLayerBelow(mLayerBelow);
	mLayerInfo.AddHeaderSize(sizeof(Header));

	// Register with layer below
	if(mLayerBelow)
	{
		if(!layerBelow->RegisterDeliverCallback(&mLayerBelowHandle, this))
		{
			gError("Unable to register with layer below.", __LINE__, __FILE__);
			return false;
		}
		if(!layerBelow->RegisterErrorCallback(&mErrorCallbackHandle, this))
		{
			gError("Unable to register with layer below for error callback.", __LINE__, __FILE__);
			return false;
		}
	}
	mNextGossip = cProtocolStack::GetTime();  // Start off scheduling right away

	// Starting from scratch
	cGroup* temp = mProtocolStack->GetLocalAddress();
	cIterator* iter = temp->GetIterator();
	mMyNode = new cPbcastNode((cEndpoint *)iter->GetData());
	if(!mMyNode)
	{
		return false;
	}
	int epBufSize = sizeof(mMyEpSer);
	if(!mMyNode->mEndpoint->Serialize(mMyEpSer, &epBufSize))
	{
		gError("Unable to fit endpoint into serialized storage buffer.", __LINE__, __FILE__);
		return false;
	}
	if(mView->Insert(mMyNode) == NULL)
	{
		gError("Unable to insert my node into view.", __LINE__, __FILE__);
		return false;
	}
	return true;
}

/*
 * cPbcastLayer::Cleanup()
 *
 * Purpose:	Cleans up the pbcast layer.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer is no longer useable.
 * Return:	true if success, else false.
 */
bool cPbcastLayer::Cleanup()
{
	if(mLayerBelow)   
	{ 
		mLayerBelow->UnregisterDeliverCallback(mLayerBelowHandle); 
		mLayerBelow->UnregisterErrorCallback(mErrorCallbackHandle);
	}
	mParam.mMcastEp->Release();	// Release the multicast endpoint

	// Delete the register list.
	if(mRegisterList) { delete mRegisterList; mRegisterList = NULL; }

	// Delete view list.
	if(mViewList)	  { delete mViewList; mViewList = NULL; }

	// Delete error callback list.
	if(mErrorCallbackList)	{ delete mErrorCallbackList; mErrorCallbackList = NULL; }

	// Clean up groups, etc.
	if(mSubnetTree)		{ delete mSubnetTree;  mSubnetTree = NULL;  }
	if(mGossipGroup)	{ delete mGossipGroup; mGossipGroup = NULL; }
	if(mSendGroup)		{ delete mSendGroup;   mSendGroup = NULL;	}
	if(mRetransReqGroup){ delete mRetransReqGroup; }
	if(mView)			{ delete mView; mView = NULL; }
	if(mFailedNodes)	{ delete mFailedNodes; mFailedNodes = NULL; }
	if(mMyNode)			{ //mMyNode->mEndpoint->Release(); 
						  mMyNode->Release(); }
	if(mMsgQueue)		{ delete mMsgQueue; }
	return true;
}

/*
 * cPbcastLayer::Send()
 *
 * Purpose:	Prints out the msg to send and passes it to layer below.
 * IN:		dest		-> The message destination.
 *			buffer		-> the actual message.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer is ready to rock.
 * Return:	true if success, else false.
 */
bool cPbcastLayer::Send(cGroup* dest, cMsgBuffer* buffer, int messageType)	
{
	bool			retVal;
	Header			header;
	SubnetInfo		sinfo;
	unsigned int	root;
	unsigned int	mySubnetIndex;
	int				current;
	cEndpoint*		outEp;

	if(!mLayerBelow)
	{
		return false;
	}

	// Send a non-pbcast message	< allows layers to sneak their own messages through >
	if((messageType != MSG_TYPE_STANDARD) && (messageType != MSG_TYPE_PBCAST))
	{
		header.msgType = PBCAST_MSG_TYPE_THRU;
		if(!buffer->AddHeader(&header, sizeof(Header)))
		{
			return false;
		}
		retVal = mLayerBelow->Send(dest, buffer, messageType);
		buffer->RemoveHeader(sizeof(Header));
		return retVal;
	}

	mParam.mPbcastsSent++;

	// Add the header
	mSeqNum++;
	header.info.data.dataSeqNum = htonl(mSeqNum);
	memcpy(header.info.data.originator, mMyEpSer, sizeof(header.info.data.originator));

	if(mSubnetTree->GetNumElements() > 1)
	{
		// Start random tree dissemination
		mSendGroup->Clear();
		mSubnetTree->GetRandomSubnet(&sinfo, &root);
		header.msgType = PBCAST_MSG_TYPE_DATA_DIRECT;
		header.info.data.subnetTreeRoot = htons((u_short)root);
		header.info.data.ttl = mSubnetTree->GetHeight(mParam.mSubnetTreeFanout);
		mySubnetIndex = mSubnetTree->Find(mMyNode->mEndpoint);
		if(mySubnetIndex != root)
		{
			// Add the route to my unicast list.
			if(sinfo.lastGossipOnSubnet)
			{
				mSendGroup->AddEndpoint(sinfo.lastGossipOnSubnet);
			}
			else
			{
				// Find the first node on the given subnet.
				if(mView->FindEpOnSubnet(sinfo.subnet, &outEp))
				{
					mSendGroup->AddEndpoint(outEp);
				}
			}
		}

		// Disseminate to my children (I could be the root)
		for(unsigned int i = 1; i < mParam.mSubnetTreeFanout+1; i++)
		{
			current = mSubnetTree->GetChild(root, mySubnetIndex, i, mParam.mSubnetTreeFanout);
			if(current == -1)
			{
				break;
			}
			sinfo = mSubnetTree->GetSubnetInfo(current);
			if(sinfo.lastGossipOnSubnet)
			{
				mSendGroup->AddEndpoint(sinfo.lastGossipOnSubnet);
			}
			else
			{
				// Find the first node on the given subnet.
				if(mView->FindEpOnSubnet(sinfo.subnet, &outEp))
				{
					mSendGroup->AddEndpoint(outEp);
				}
			}
		}
		buffer->AddHeader(&header, sizeof(Header));
		mLayerBelow->Send(mSendGroup, buffer, MSG_TYPE_UNICAST);
		buffer->RemoveHeader(sizeof(Header));
	}

	// Multicast to local subnet.
	header.msgType = PBCAST_MSG_TYPE_DATA;
	buffer->AddHeader(&header, sizeof(Header));
	mSendGroup->Clear();
	mSendGroup->AddEndpoint(mParam.mMcastEp);
	mLayerBelow->Send(mSendGroup, buffer, MSG_TYPE_MULTICAST);
	buffer->RemoveHeader(sizeof(Header)); // Must remove my header after send!

	// Insert into my own message buffer!
	MsgInfo info;
	info.retransReqRemaining = 0;
	info.msg    = buffer;
	info.node   = mMyNode;
	info.seqNum = mSeqNum;	// No need to net order it.
	if(!mMsgQueue->Insert(info))
	{
		gError("Unable to insert msg i'm sending into msg queue!", __LINE__, __FILE__);
	}
	return true; 
}

/*
 * cPbcastLayer::RegisterDeliverCallback()
 *
 * Purpose:	Registers the deliver callback.
 * IN:		callback	-> The callback function to register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is registered.
 * Return:	The handle used for unregistering.
 */
bool cPbcastLayer::RegisterDeliverCallback(cHandle* handle, cDeliver* callback) 
{ 
	handle->mLayer = this;
	return mRegisterList->AddObject(handle, (cObject *)callback);
}

/*
 * cPbcastLayer::UnregisterDeliverCallback()
 *
 * Purpose:	Unregisters the deliver callback.
 * IN:		handle	-> The handle that was received at register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is removed.
 * Return:	true if success, else false.
 */
bool cPbcastLayer::UnregisterDeliverCallback(cHandle handle)	
{ 
	if(handle.mLayer == this)
	{
		return mRegisterList->RemoveObject(handle);
	}
	else if(mLayerBelow)
	{
		return mLayerBelow->UnregisterDeliverCallback(handle);
	}
	else
	{
		return false;
	}
}

/*
 * cPbcastLayer::RegisterErrorCallback()
 *
 * Purpose:	Registers the asynchronous error callback.
 * IN:		callback	-> The callback function to register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is registered.
 * Return:	The handle used for unregistering.
 */
bool cPbcastLayer::RegisterErrorCallback(cHandle* handle, cErrorCallback* callback)
{ 
	handle->mLayer = this;
	return mErrorCallbackList->AddObject(handle, (cObject *)callback);
}

/*
 * cPbcastLayer::UnregisterErrorCallback()
 *
 * Purpose:	Unregisters the error callback.
 * IN:		handle	-> The handle that was received at register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is removed.
 * Return:	true if success, else false.
 */
bool cPbcastLayer::UnregisterErrorCallback(cHandle handle)
{ 
	if(handle.mLayer == this)
	{
		return mErrorCallbackList->RemoveObject(handle);
	}
	else if(mLayerBelow)
	{
		return mLayerBelow->UnregisterErrorCallback(handle);
	}
	else
	{
		return false;
	}
}

/*
 * cPbcastLayer::Schedule()
 *
 * Purpose:	This is where pbcast does most of its dirty work
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if success, else false.
 */
bool cPbcastLayer::Schedule()
{
	DWORD	thisTick;

	if(!mStarted)
	{
		_ViewChange();	// Deliver first view change.
	}

	thisTick = cProtocolStack::GetTime();
	if(thisTick > mNextGossip)
	{
		if(mView->GetNumElements() > 1)
		{
			mNextGossip = thisTick + mParam.mGossipInterval;		
		}
		else
		{
			mNextGossip = thisTick + DISCOVERY_INTERVAL;
		}

		// Reset parameters that last only for this round.
		mBytesRetransmittedInThisRound = 0;
		mNumRetransReqInThisRound = 0;

		// Attempt to deliver msgs i sent myself.
		_AttemptDelivery(mMyNode);

		_ChooseGossipGroup();
		if(mView->GetNumElements() == 1)
		{
			_FullGossip(true);
		}
		else
		{
			_MinGossip();
		}
	}
	else
	{
		mMsgQueue->Compact();
		_CheckForDeadNodes();
	}
	return true;
}

/*
 * cPbcastLayer::_CheckForDeadNodes()
 *
 * Purpose:	Checks for dead nodes and removes them.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	If there are nodes that exceed death timeout, they are removed.
 * Return:	true if success, else false.
 */
bool cPbcastLayer::_CheckForDeadNodes()
{
	cIterator*	iter;
	cPbcastNode*	resultNode;
	DWORD			now = cProtocolStack::GetTime();

	iter = mFailedNodes->GetNodeIterator();
	while(!iter->Done())
	{
		resultNode = (cPbcastNode *)iter->GetData();
		if( (now - resultNode->mLastUpdateTime) > mParam.mTimeToCleanupNode )
		{
			mMsgQueue->Clear(resultNode);	// Clear out all messages from this node.
			iter->DeleteCurrent();
			delete resultNode;								// Kill it!
		}
		else
		{
			iter->GetNext();
		}
	}
	return true;
}

/*
 * cPbcastLayer::_HandleData()
 *
 * Purpose:	Handles the receipt of a data message.
 * IN:		sender		-> the sender of the message.
 *			header		-> The message header.
 *			msg			-> The associated msg.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if success, else false.
 */
bool cPbcastLayer::_HandleData(cEndpoint* sender, Header* header, cMsgBuffer* msg)
{
	cPbcastNode*	resultNode;

	// Demarshal the header
	header->info.data.dataSeqNum = ntohl(header->info.data.dataSeqNum);

	// Make sure we want this data msg
	if(mView->Find(sender, &resultNode))
	{
		// If this is the first message, use it to set initial lastDeliveredSeqNum
		if(!resultNode->mSeenFirstMsg)
		{
			// Wait until you see a brand new message.
			if(cPbcastNode::SeqNumOlder(resultNode->mLastDeliveredSeqNum, header->info.data.dataSeqNum))
			{
				resultNode->mLastDeliveredSeqNum = header->info.data.dataSeqNum-1;
				resultNode->mSeenFirstMsg = true;
			}
			else
			{
				return true; // Not ready to receive from this dude yet, ignore.
			}
		}
		else if(cPbcastNode::SeqNumOlder(header->info.data.dataSeqNum, resultNode->mLastDeliveredSeqNum))
		{
			return true;	// This is an old message, discard.
		}
	}
	else
	{
		return true;	// Don't know about this node...ignore it
	}

	MsgInfo	info;
	info.msg = msg;
	info.node = resultNode;
	info.seqNum = header->info.data.dataSeqNum;
	info.retransReqRemaining = 0;

	// Insert latest copy of msg into the queue.
	if(!mMsgQueue->Insert(info))
	{
		gError("Unable to insert new message into message queue!", __LINE__, __FILE__);
	}

	// Attempt to deliver messages for this node.
	_AttemptDelivery(resultNode);

	return true;	// Handled
}

/*
 * cPbcastLayer::_HandleDataDissemination()
 *
 * Purpose:	Handles the receipt of a direct data message and disseminating it to more subnets.
 * IN:		sender		-> the sender of the message.
 *			header		-> The message header.
 *			msg			-> The associated msg.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if success, else false.
 */
bool cPbcastLayer::_HandleDataDissemination(cEndpoint* sender, Header* header, cMsgBuffer* msg)
{
	cEndpoint*		outEp;
	cPbcastNode*	resultNode;
	u_long			dataSeqNum;
	u_short			subnetTreeRoot;
	unsigned int	mySubnetIndex;
	unsigned int	current;
	SubnetInfo		sinfo;

	// Demarshal the header
	dataSeqNum		  = ntohl(header->info.data.dataSeqNum);
	subnetTreeRoot    = ntohs(header->info.data.subnetTreeRoot);

	// Make sure we want this data msg
	if(!mView->Find(sender, &resultNode))
	{
		return true;	// Don't know about this node...ignore it
	}
	else
	{
		// If this is the first message, use it to set initial lastDeliveredSeqNum
		if(!resultNode->mSeenFirstMsg)
		{
			// Wait until see a brand new message.
			if(cPbcastNode::SeqNumOlder(resultNode->mLastDeliveredSeqNum, dataSeqNum))
			{
				resultNode->mLastDeliveredSeqNum = dataSeqNum-1;
				resultNode->mSeenFirstMsg = true;
			}
		}
	}

	MsgInfo	info;
	info.msg = msg;
	info.node = resultNode;
	info.seqNum = dataSeqNum;
	info.retransReqRemaining = 0;

	// Insert latest copy of msg into the queue.
	if(!mMsgQueue->Insert(info))
	{
		gError("Unable to insert new message into message queue!", __LINE__, __FILE__);
	}

	// Disseminate to other subnets in tree (if necessary)
	if(header->info.data.ttl)
	{
		header->info.data.ttl--;

		// Send this baby on.
		mGossipGroup->Clear();
		mySubnetIndex = mSubnetTree->Find(mMyNode->mEndpoint);
		for(unsigned int i = 1; i < mParam.mSubnetTreeFanout+1; i++)
		{
			current = mSubnetTree->GetChild(subnetTreeRoot, mySubnetIndex, i, mParam.mSubnetTreeFanout);
			if(current == -1)
			{
				break;
			}
			sinfo = mSubnetTree->GetSubnetInfo(current);
			if(sender->IsSameSubnet(sinfo.subnet))
			{
				continue;	// Don't send to the originator's subnet!
			}
			if(sinfo.lastGossipOnSubnet)
			{
				mGossipGroup->AddEndpoint(sinfo.lastGossipOnSubnet);
			}
			else
			{
				// Find the first node on the given subnet.
				if(mView->FindEpOnSubnet(sinfo.subnet, &outEp))
				{
					mGossipGroup->AddEndpoint(outEp);
				}
			}
		}
		msg->AddHeader(header, sizeof(Header));
		mLayerBelow->Send(mGossipGroup, msg, MSG_TYPE_UNICAST);
		msg->RemoveHeader(sizeof(Header));
	}

	// Multicast to my local subnet
	mGossipGroup->Clear();
	mGossipGroup->AddEndpoint(mParam.mMcastEp);
	header->msgType = PBCAST_MSG_TYPE_DATA;
	msg->AddHeader(header, sizeof(Header));
	mLayerBelow->Send(mGossipGroup, msg, MSG_TYPE_MULTICAST);
	msg->RemoveHeader(sizeof(Header));

	// Attempt to deliver messages for this node.
	_AttemptDelivery(resultNode);
	return true;	// Handled
}

/*
 * cPbcastLayer::_AttemptDelivery()
 *
 * Purpose:	Attempts to deliver messages for this node if possible.
 * IN:		node		-> The node to attempt delivery for.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if success, else false.
 */
bool cPbcastLayer::_AttemptDelivery(cPbcastNode* node)
{
	MsgInfo	info;
	info.msg = NULL;
	info.node = node;
	info.retransReqRemaining = 0;
	info.seqNum = node->mLastDeliveredSeqNum + 1;

	while(1)
	{
		if(!mMsgQueue->Find(&info))
		{
			break;	// Next sequence to deliver is not in the queue.
		}
		if(info.retransReqRemaining > 0)
		{
			break;	// Awaiting retransmit of this message.
		}

		// Deliver to callbacks.
		cIterator* iter;
		cDeliver*  deliver;
		iter = mRegisterList->GetIterator();
		while(!iter->Done())
		{
			deliver = (cDeliver *)iter->GetData();
			if(info.msg)
			{
				deliver->Deliver(node->mEndpoint, info.msg, MSG_TYPE_STANDARD);
			}
			else
			{
				_DeliverErrorCallback(NULL, ASYNCH_ERROR_MSG_GAP);
				deliver->Deliver(node->mEndpoint, NULL, MSG_TYPE_NULL);
			}
			iter->GetNext();
		}
		node->mLastDeliveredSeqNum++;
		info.seqNum++;
	}
	return true;
}

/*
 * cPbcastLayer::_HandleRetrReq()
 *
 * Purpose:	Handles the receive of a request for message retransmit.
 * IN:		sender		-> The sender of the message.
 *			header		-> The message header.
 *			msg			-> The associated msg.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if success, else false.
 */
bool cPbcastLayer::_HandleRetrReq(cEndpoint* sender, Header* header, cMsgBuffer* msg)
{
	unsigned char*	digest;
	int				digestSize;
	u_short			index;
	char*			payload;
	int				size;
	int				bufSize;
	unsigned int	amtRetransmit;
	cPbcastNode*	node;
	u_long			seqNum;
	MsgInfo			info;
	Header			retransHeader;

	// Demarshal the header
	header->info.retrans.numRequests = ntohl(header->info.retrans.numRequests);
	header->info.retrans.senderGossipSeqNum = ntohl(header->info.retrans.senderGossipSeqNum);

	if(header->info.retrans.senderGossipSeqNum != mMyNode->mGossipSeqNum)
	{
		gInfo("Recv'd old retransmission request...IGNORING.", __LINE__, __FILE__);
		return true;
	}
	
	gInfo("Recv'd retransmission request.", __LINE__, __FILE__);
	msg->GetPayload((void **)&payload, &size);

	mView->GetDigest(&digest, &digestSize);
	if(memcmp(payload, digest, digestSize) != 0)
	{
		gInfo("Recv'd retransmission request with wrong digest...IGNORING.", __LINE__, __FILE__);
		return true;
	}
	payload += digestSize;

	// Process requests
	for(unsigned int i = 0; i < header->info.retrans.numRequests; i++)
	{
		memcpy(&index, payload, sizeof(index));
		payload += sizeof(index);
		index = ntohs(index);	// Unmangle.
		memcpy(&seqNum, payload, sizeof(u_long));
		payload += sizeof(u_long);
		seqNum = ntohl(seqNum);
		node = mView->GetNode(index);

		info.node = node;
		info.seqNum = seqNum;
		if(mMsgQueue->Find(&info))
		{
			if(info.msg == NULL)
			{
				// We had to skip this message ourselves.
				continue;
			}
			amtRetransmit = mBytesRetransmittedInThisRound + info.msg->GetPayloadSize();
			if( amtRetransmit > mParam.mMaxBytesRetransmitPerRound )
			{
				return true;	// handled.
			}
			mBytesRetransmittedInThisRound = amtRetransmit;

			// Retransmit the message.
			retransHeader.msgType = PBCAST_MSG_TYPE_RETR;
			retransHeader.info.data.dataSeqNum = htonl(seqNum);
			bufSize = sizeof(retransHeader.info.data.originator);
			if(!node->mEndpoint->Serialize(&(retransHeader.info.data.originator[0]), &bufSize))
			{
				gError("Unable to serialize a node's endpoint into retransmission header.", __LINE__, __FILE__);
				return false;
			}

			if(!info.msg->AddHeader(&retransHeader, sizeof(Header)))
			{
				return false;
			}
			mRetransReqGroup->Clear();
			mRetransReqGroup->AddEndpoint(sender);
			info.msg->AddRef();
			mLayerBelow->Send(mRetransReqGroup, info.msg, MSG_TYPE_UNICAST);
			info.msg->RemoveHeader(sizeof(Header));	// Must remove my header!
			info.msg->Release();
		}
	}
	return true;
}

/*
 * cPbcastLayer::_HandleMinGossip()
 *
 * Purpose:	Handles the receipt of a minimal gossip message.
 * IN:		header		-> The message header.
 *			msg			-> The associated msg.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if success, else false.
 */
bool cPbcastLayer::_HandleMinGossip(cEndpoint* sender, Header* header, cMsgBuffer* msg)
{
	char*			payload;
	int				size;
	cPbcastNode		node;
	cPbcastNode*	resultNode;
	DWORD			now = cProtocolStack::GetTime();
	unsigned char*	digest;
	int				digestSize;

	mParam.mMinGossipsReceived++;

	// Demarshal the header
	header->info.gossip.numViewNodes = ntohl(header->info.gossip.numViewNodes);
	header->info.gossip.gossipSeqNum = ntohl(header->info.gossip.gossipSeqNum);

	// Make sure this is a gossip we want.
	if(mView->Find(sender, &resultNode))
	{
		if(cPbcastNode::SeqNumOlder(resultNode->mGossipSeqNum, header->info.gossip.gossipSeqNum))
		{
			resultNode->mGossipSeqNum = header->info.gossip.gossipSeqNum;
		}
		else
		{
			return true;	// Gossip already handled
		}
		mSubnetTree->UpdateGossip(resultNode->mEndpoint);	// Set last node I exchanged gossip with.
	}

	gInfo("Recv'd a gossip message.", __LINE__, __FILE__);

	msg->GetPayload((void **)&payload, &size);

	// Compare digests
	mView->GetDigest(&digest, &digestSize);
	if( (header->info.gossip.numViewNodes != (unsigned)mView->GetNumElements()) ||
		(memcmp(payload, digest, digestSize) != 0) )
	{
		// Send other party a full gossip.
		mGossipGroup->Clear();
		mGossipGroup->AddEndpoint(sender);
		_FullGossip(true);	// request a retransmit
		return true;
	}
	payload += digestSize;

	// Digest the same -> view is the same.  process.
	for(unsigned int i = 0; i < (unsigned)mView->GetNumElements(); i++)
	{
		resultNode = mView->GetNode(i);
		payload = cPbcastNode::MinDeserialize(&node, payload, &size);
		if(resultNode->HeartbeatOlder(&node))
		{
			resultNode->mHeartbeat = node.mHeartbeat;
			resultNode->mLastUpdateTime = now;
		}
		if(cPbcastNode::SeqNumOlder(resultNode->mLastDeliveredSeqNum, node.mLastDeliveredSeqNum))
		{
			_AddToRetransmitReqMsg(i, resultNode, &node);	// Add retransmission requests for node's msgs.
		}
	} /* end for */
	
	// Send any necessary retransmit message.
	if(mNumRetranReq)
	{
		cMsgBuffer* retrans;
		Header		rHeader;

		retrans = cBufferManager::GetBuffer();
		if(!retrans->Init((mRetransmitPtr-&mRetransmitBuffer[0])+mLayerInfo.GetCumHeaderSize(), mLayerInfo.GetCumHeaderSize()))
		{
			gError("Unable to initialize message buffer for send.", __LINE__, __FILE__);
		}
		retrans->SetPayload(&mRetransmitBuffer, (mRetransmitPtr-&mRetransmitBuffer[0]));

		// Add header to msg.
		rHeader.msgType = PBCAST_MSG_TYPE_RETR_REQ;
		rHeader.info.retrans.numRequests = htonl(mNumRetranReq);
		rHeader.info.retrans.senderGossipSeqNum = htonl(header->info.gossip.gossipSeqNum);
		if(!retrans->AddHeader(&rHeader, sizeof(Header)))
		{
			gError("Unable to add header.", __LINE__, __FILE__);
		}
		mRetransReqGroup->Clear();
		mRetransReqGroup->AddEndpoint(sender);
		retrans->AddRef();
		mLayerBelow->Send(mRetransReqGroup, retrans, MSG_TYPE_UNICAST);
		retrans->RemoveHeader(sizeof(Header)); // Must remove my header.
		retrans->Release();
		_ResetRetransmitReqMsg();
	}

	// Clean up node?
	node.~cPbcastNode();
	return true;
}

/*
 * cPbcastLayer::_HandleFullGossip()
 *
 * Purpose:	Handles the receive of a complete gossip message.
 * IN:		header		-> The message header.
 *			msg			-> The associated msg.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if success, else false.
 */
bool cPbcastLayer::_HandleFullGossip(cEndpoint* sender, Header* header, cMsgBuffer* msg)
{
	char*			payload;
	int				size;
	cPbcastNode		node;
	cPbcastNode*	resultNode;
	DWORD			now = cProtocolStack::GetTime();
	bool			bNewView = false;

	mParam.mFullGossipsReceived++;

	// Demarshal the header
	header->info.gossip.numViewNodes = ntohl(header->info.gossip.numViewNodes);
	header->info.gossip.gossipSeqNum = ntohl(header->info.gossip.gossipSeqNum);

	// Make sure this is a gossip we want.
	if(mView->Find(sender, &resultNode))
	{
		if(cPbcastNode::SeqNumOlder(resultNode->mGossipSeqNum, header->info.gossip.gossipSeqNum))
		{
			resultNode->mGossipSeqNum = header->info.gossip.gossipSeqNum;
		}
		else
		{
			return true;	// Gossip already handled
		}
	}

	gInfo("Recv'd a full gossip message.", __LINE__, __FILE__);

	msg->GetPayload((void **)&payload, &size);

	// Process the other dude's view
	for(unsigned int i = 0; i < header->info.gossip.numViewNodes; i++)
	{
		payload = node.Deserialize(payload, &size);
		if(mView->Find(node.mEndpoint, &resultNode))	// Update node in our view.
		{
			if(resultNode->HeartbeatOlder(&node))
			{
				resultNode->mHeartbeat = node.mHeartbeat;
				resultNode->mLastUpdateTime = now;
			}
		}
		else if(mFailedNodes->Find(node.mEndpoint, &resultNode))  // Check on node in failed list.
		{
			continue;	// Ignore nodes that have failed!
		}
		else	// New node...add it.
		{
			resultNode = new cPbcastNode(node.mEndpoint);
			if(!resultNode)
			{
				gError("Unable to allocate new pbcast node to add to view!", __LINE__, __FILE__);
			}
			else
			{
				if(!mView->Insert(resultNode))
				{
					gError("Unable to add new node to the view!", __LINE__, __FILE__);
					resultNode->Release();
					continue;
				}
				resultNode->mHeartbeat = node.mHeartbeat;
				resultNode->mLastUpdateTime = now;
				resultNode->mLastDeliveredSeqNum = node.mLastDeliveredSeqNum; // Start reasonable.

				// Add endpoint's subnet, if necessary.
				if(!mSubnetTree->Insert(resultNode->mEndpoint))
				{
					gError("Unable to add possible new subnet to subnet tree.", __LINE__, __FILE__);
				}
			}
			bNewView = true;
		}
	} /* end for */

	// Report view change if necessary.
	if(bNewView)
	{
		_ViewChange();
	}

	// Clean up node?
	node.~cPbcastNode();
	return true;
}

/*
 * cPbcastLayer::_ResetRetransmitReqMsg()
 *
 * Purpose:	Resets the message used to request retranmissions.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The retransmission buffer is reset.
 * Return:	true if success, else false.
*/
bool cPbcastLayer::_ResetRetransmitReqMsg()
{
	mRetransmitBytesLeft = sizeof(mRetransmitBuffer);
	mRetransmitPtr		 = &mRetransmitBuffer[0];
	mNumRetranReq		 = 0;

	// Add the view digest to the front.
	unsigned char*  digest;
	int				digestSize;
	mView->GetDigest(&digest, &digestSize);
	memcpy(mRetransmitPtr, digest, digestSize);
	mRetransmitPtr += digestSize;
	mRetransmitBytesLeft -= digestSize;

	return true;
}

/*
 * cPbcastLayer::_AddToRetransmitReqMsg()
 *
 * Purpose:	Adds any retransmit messages it can to the retransmit msg.
 * IN:		
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if success, else false.
*/
bool cPbcastLayer::_AddToRetransmitReqMsg(u_short viewIndex, cPbcastNode* localNode, cPbcastNode* remoteNode)
{
	MsgInfo rInfo;

	viewIndex = htons(viewIndex);	// Mangle-to-network-order
	rInfo.node = localNode;
	unsigned int gap = cPbcastNode::SeqNumDifference(localNode->mLastDeliveredSeqNum, remoteNode->mLastDeliveredSeqNum);
	if(gap > MAX_MSG_GAP)
	{
		_DeliverErrorCallback(NULL, ASYNCH_ERROR_MSG_GAP);
		localNode->mLastDeliveredSeqNum = remoteNode->mLastDeliveredSeqNum;
		localNode->mSeenFirstMsg = false;	// Start over (should i?)
		return true;
	}

	// Loop until seqNum is not older than other node's last delivered plus one
	for(u_long seqNum = localNode->mLastDeliveredSeqNum+1; cPbcastNode::SeqNumOlder(seqNum, remoteNode->mLastDeliveredSeqNum+1); seqNum++)
	{
		rInfo.seqNum = seqNum;
		rInfo.retransReqRemaining = 0;
		rInfo.msg = NULL;

		// Break out of loop if exceeded maximum retransmit requests.
		if((unsigned)mRetransmitBytesLeft < (sizeof(viewIndex)+sizeof(u_long))) break;	// Done all we can here!

		// See if we already sent a retrans req.
		if(mMsgQueue->Find(&rInfo))
		{
			if(rInfo.retransReqRemaining > 0)
			{
				// Haven't sent too many so far.
				rInfo.retransReqRemaining--;
				mMsgQueue->Replace(&rInfo);
			}
			else
			{
				// Either sent too many retransmit, or msg in queue out of order.								
				continue;
			}
		}
		else
		{
			// Make sure don't exceed maximum number of msgs can ask for retransmit.
			if(mNumRetransReqInThisRound >= mParam.mMaxRetransRequestsPerRound)
			{
				return true;
			}
			mNumRetransReqInThisRound++;

			// This is the first time requesting retrans for this.
			rInfo.retransReqRemaining = mParam.mNumRetransmitRequestAttempts;
			if(!mMsgQueue->Insert(rInfo))
			{
				gError("Unable to insert NULL retransmit msg into msg queue.", __LINE__, __FILE__);
			}
		}

		// Fill in actual retransmit information.
		mParam.mNumRetansReq++;
		memcpy(mRetransmitPtr, &viewIndex, sizeof(viewIndex));	// Copy index
		mRetransmitPtr += sizeof(viewIndex);
		seqNum = htonl(seqNum);
		memcpy(mRetransmitPtr, &seqNum, sizeof(u_long));		// Copy desired seqNum
		seqNum = ntohl(seqNum);
		mRetransmitPtr += sizeof(u_long);
		mRetransmitBytesLeft -= (sizeof(viewIndex) + sizeof(u_long));
		mNumRetranReq++;
	}
	return true;
}

/*
 * cPbcastLayer::_MinGossip()
 *
 * Purpose:	Sends a minimal gossip to the gossip group.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if success, else false.
 */
void cPbcastLayer::_MinGossip()
{
	cMsgBuffer*		msg;
	cIterator*		iter;
	bool			bNewView = false;
	int				size;
	char*			payload;
	Header			header;
	cPbcastNode	   *pbnode;
	DWORD			now = cProtocolStack::GetTime();
	unsigned char  *digest;
	char*			digestSpot;
	int				digestSize;

	mMyNode->mHeartbeat++;	// Increase my heartbeat
	mParam.mMinGossipsSent += mParam.mFanout;

	mView->GetDigest(&digest, &digestSize);

	// Set up the payload
	size = digestSize + (mView->GetNumElements() * (sizeof(mMyNode->mHeartbeat) + sizeof(mMyNode->mLastDeliveredSeqNum)));
	msg = cBufferManager::GetBuffer();
	msg->AddRef();
	if(!msg->Init(size+mLayerInfo.GetCumHeaderSize(), mLayerInfo.GetCumHeaderSize()))
	{
		gError("Unable to initialize message buffer for send.", __LINE__, __FILE__);
	}
	msg->GetPayload((void **)&payload, &size);

	// Reserve space for the digest.
	digestSpot = payload;
	payload += digestSize;

	// Send view nodes.
	iter = mView->GetNodeIterator();
	while(!iter->Done())
	{
		pbnode = (cPbcastNode *)iter->GetData();
		if(pbnode != mMyNode)
		{
			if((now - pbnode->mLastUpdateTime) > mParam.mTimeToFailNode)
			{
				mFailedNodes->Insert(pbnode);	// Add to failed nodes
				iter->DeleteCurrent();			// Delete from view
				bNewView = true;
				continue;
			}
		}	
		payload = cPbcastNode::MinSerialize(pbnode, payload, &size);
		if(!payload)
		{
			gError("Unable to serialize a pbnode for gossip.", __LINE__, __FILE__);
		}
		iter->GetNext();
	}	

	// Report new view.
	if(bNewView)
	{
		_ViewChange();
		mView->GetDigest(&digest, &digestSize);
	}

	// Add header to msg.
	mMyNode->mGossipSeqNum++;
	header.msgType = PBCAST_MSG_TYPE_MIN_GOSSIP;
	header.info.gossip.gossipSeqNum   = htonl(mMyNode->mGossipSeqNum);
	header.info.gossip.numViewNodes   = htonl((u_long)mView->GetNumElements());
	if(!msg->AddHeader(&header, sizeof(Header)))
	{
		gError("Unable to add header.", __LINE__, __FILE__);
	}

	// Add digest
	memcpy(digestSpot, digest, digestSize);

	// Send gossip to chosen group.
	if(mView->GetNumElements() < 2)
	{
		mLayerBelow->Send(mGossipGroup, msg, MSG_TYPE_BCAST);
	}
	else
	{
		mLayerBelow->Send(mGossipGroup, msg, MSG_TYPE_UNICAST);
	}
	msg->RemoveHeader(sizeof(Header));	// Must remove my header.
	msg->Release();
}

/*
 * cPbcastLayer::_FullGossip()
 *
 * Purpose:	Sends a full gossip to the gossip group.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if success, else false.
 */
void cPbcastLayer::_FullGossip(bool request)
{
	cMsgBuffer*		msg;
	cIterator*		iter;
	bool			bNewView = false;
	int				size;
	char*			payload;
	Header			header;
	cPbcastNode	   *pbnode;
	DWORD			now = cProtocolStack::GetTime();

	mParam.mFullGossipsSent += mGossipGroup->GetNumElements();
	mMyNode->mHeartbeat++;	// Increase my heartbeat

	// Set up the payload
	size = mMyNode->GetSize() * mView->GetNumElements();
	msg = cBufferManager::GetBuffer();
	msg->AddRef();
	if(!msg->Init(size+mLayerInfo.GetCumHeaderSize(), mLayerInfo.GetCumHeaderSize()))
	{
		gError("Unable to initialize message buffer for send.", __LINE__, __FILE__);
	}
	msg->GetPayload((void **)&payload, &size);

	// Send view nodes.
	iter = mView->GetNodeIterator();
	while(!iter->Done())
	{
		pbnode = (cPbcastNode *)iter->GetData();
		if(pbnode != mMyNode)
		{
			if((now - pbnode->mLastUpdateTime) > mParam.mTimeToFailNode)
			{
				mFailedNodes->Insert(pbnode);	// Add to failed nodes
				iter->DeleteCurrent();			// Delete from view
				bNewView = true;
				continue;
			}
		}
		payload = pbnode->Serialize(payload, &size);
		if(!payload)
		{
			gError("Unable to serialize a pbnode for gossip.", __LINE__, __FILE__);
		}
		iter->GetNext();
	}	

	// Add header to msg.
	mMyNode->mGossipSeqNum++;
	if(request)
	{
		header.msgType = PBCAST_MSG_TYPE_FULL_GOSSIP_REQ;
	}
	else
	{
		header.msgType = PBCAST_MSG_TYPE_FULL_GOSSIP;
	}
	header.info.gossip.gossipSeqNum   = htonl(mMyNode->mGossipSeqNum);
	header.info.gossip.numViewNodes   = htonl((u_long)mView->GetNumElements());
	if(!msg->AddHeader(&header, sizeof(Header)))
	{
		gError("Unable to add header.", __LINE__, __FILE__);
	}

	// Send gossip to chosen group.
	if(mView->GetNumElements() < 2)
	{
		mLayerBelow->Send(mGossipGroup, msg, MSG_TYPE_BCAST);
	}
	else
	{
		mLayerBelow->Send(mGossipGroup, msg, MSG_TYPE_UNICAST);
	}
	msg->RemoveHeader(sizeof(Header));	// Must remove my header.
	msg->Release();

	// Report new view.
	if(bNewView)
	{
		_ViewChange();
	}
}

/*
 * cPbcastLayer::_ChooseGossipGroup()
 *
 * Purpose:	Chooses endpoints to gossip to.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	Parameters should be ready to rock.
 * Return:	true if success, else false.
 */
bool cPbcastLayer::_ChooseGossipGroup()
{
	cEndpoint*		ep;
	cPbcastNode*	node;
	unsigned int	total;

	mGossipGroup->Clear();
	if(mView->GetNumElements() > 1)
	{
		// Send to individual nodes.
		if((mView->GetNumElements()-1) > (signed)mParam.mFanout)
		{
			total = mParam.mFanout;
		}
		else
		{
			total = mView->GetNumElements()-1;
		}

		// Fill gossip group with 'total' distinct nodes.
		for(unsigned int i = 0; i < total; i++)
		{
			node = mMyNode;
			while( (node == mMyNode) || (mGossipGroup->IsMember(node->mEndpoint)) )
			{
				node = mView->GetRandomNode();			
			}
			mGossipGroup->AddEndpoint(node->mEndpoint);
		}
	}
	else
	{
		// Send to subnets.
		if(mSubnetTree->GetNumElements() > (signed)mParam.mFanout)
		{
			total = mParam.mFanout;
		}
		else
		{
			total = mSubnetTree->GetNumElements();
		}


		cIPEndpoint* iep;

		// Get random subnets to gossip with.
		for(unsigned int i = 0; i < total; i++)
		{
			do
			{
				ep = mSubnetTree->GetRandomSubnet();
			} while(mGossipGroup->IsMember(ep));
			iep = (cIPEndpoint*)ep;
			mGossipGroup->AddEndpoint(ep);
		}
	}
	return true;
}

/*
 * cPbcastLayer::_Pinf()
 *
 * Purpose:	Probability of infection in a given round.
 * NOTE:	*See Appendix A of "A Gossip Style Failure Detection Service", van Renesse, Minsky, Hayden
 * IN:		k		-> Number of currently infected members.
 * Return:	double, probability that a non-infected member is infected in a round, given k infected.
 */
inline double cPbcastLayer::_Pinf(double k)
{
	double n = mView->GetNumElements();
	double Parrival = 1.0 - 0.0005;
	return 1 - pow((1 - (Parrival / (n-1))), k);
}

/*
 * cPbcastLayer::_K()
 *
 * Purpose:	Computes the number of infected members in round i (assuming K(i-1)) infected members.
 * NOTE:	*See Appendix A of "A Gossip Style Failure Detection Service", van Renesse, Minsky, Hayden
 * IN:		i	-> The current round.
 * Return:	number of infected members in this round; k(0) = 1.0
 */
double cPbcastLayer::_K(double i)
{
	double n = mView->GetNumElements();

	if(i == 0) { return 1.0; }
	double prev = _K(i-1);
	return prev + (n-1.0-prev)*_Pinf(prev);
}

/*
 * cPbcastLayer::_ComputeParams()
 *
 * Purpose:	Computes the pbcast parameters (based on old ones and current situation).
 * NOTE:	*See "A Gossip Style Failure Detection Service", van Renesse, Minsky, Hayden
 * PostCnd:	Parameters should be ready to rock.
 * Return:	true if success, else false.
 */
bool cPbcastLayer::_ComputeParams()
{
	// Compute new gossip interval
	double	gossipSize;
	double	gossipInterval;

	gossipSize = (double) ( DIGEST_SIZE + 
				            mLayerInfo.GetCumHeaderSize() + 
				            (mView->GetNumElements()*cPbcastNode::GetMinSerialSize()) );

	gossipSize = gossipSize * 1000;						// to convert to ms easier.
	gossipInterval = gossipSize/mParam.mMaxBandwidth;	// yields ms/message
	mParam.mGossipInterval = (DWORD)gossipInterval;

	// Compute the new fanout
	if(mParam.mGossipInterval < MIN_GOSSIP_INTERVAL)
	{
		mParam.mFanout = MIN_GOSSIP_INTERVAL / mParam.mGossipInterval;
		mParam.mGossipInterval = MIN_GOSSIP_INTERVAL;
		if(mParam.mFanout > MAX_FANOUT)
		{
			mParam.mFanout = MAX_FANOUT;
		}
		if(!mParam.mFanout)
		{
			mParam.mFanout = 1;
		}
	}
	else
	{
		mParam.mFanout = 1;
	}

	// Compute old time to cleanup.
	mParam.mTimeToCleanupNode = 2*mParam.mTimeToFailNode;
	
	// Compute new time to cleanup messages
	mParam.mTimeToCleanupMsg = 2*mParam.mTimeToFailNode;	// to be safe
	mMsgQueue->SetCleanupTime(mParam.mTimeToCleanupMsg);

	// Compute new time to suspect failure 
	double r = 0;
	if(mView->GetNumElements() < 50)
	{
		r = (3.2 * log(mView->GetNumElements()) + 17.5); // << From Dan, James, Jed's work on pbcast >>
	}
	else
	{
		double pmistake;
		do
		{
			r++;
			pmistake = (mView->GetNumElements()-1)*(1-pow((_K(r)/(mView->GetNumElements()-1)),(mView->GetNumElements()-1)));
		} while(pmistake > mParam.mProbFailMistake);
	}
	mParam.mTimeToFailNode = (DWORD) ( (double)mParam.mGossipInterval * r);
	return true;
}

/*
 * cPbcastLayer::_SetupParams()
 *
 * Purpose:	Initializes all of the pbcast parameters
 * IN:		param		-> incoming parameters
 * OUT:		-
 * Cond:	-
 * PostCnd:	Parameters should be ready to rock.
 * Return:	true if success, else false.
 */
bool cPbcastLayer::_SetupParams(cPbcastLayerParam *param)
{
	cIterator*	iter;

	// Set everything initially to zero.
	memset(&mParam, 0, sizeof(mParam));

	mParam.mProbFailMistake = param->mProbFailMistake;
	mParam.mNumRetransmitRequestAttempts = param->mNumRetransmitRequestAttempts-1;
	mParam.mFanout = 3;		// Large initial fanout for quicker discovery.
	mParam.mGossipInterval = SCHEDULE_INTERVAL;
	mParam.mMaxBytesRetransmitPerRound = param->mMaxBytesRetransmitPerRound;
	mParam.mMaxRetransRequestsPerRound = param->mMaxRetransRequestsPerRound;
	mParam.mTimeToFailNode	  = T_FAIL;
	mParam.mTimeToCleanupNode = T_CLEANUP;
	mParam.mTimeToCleanupMsg  = 2*T_CLEANUP;
	mParam.mMaxBandwidth	  = param->mMaxBandwidth;
	mParam.mSubnetTreeFanout  = SUBNET_TREE_FANOUT;
	mParam.mMcastEp			  = param->mMulticastEndpoint;
	mParam.mMcastEp->AddRef();

	// Set up the view group
	mView = new cPbcastView(16, 4);
	if(!mView)
	{
		return false;
	}
	mFailedNodes = new cPbcastView(4, 2);
	if(!mFailedNodes)
	{
		return false;
	}
	mGossipGroup = new cGroup(MAX_FANOUT);
	if(!mGossipGroup)
	{
		return false;
	}

	// Set up subnet group
	mSubnetTree = new cSubnetTree(4, 1);
	if(!mSubnetTree)
	{
		return false;
	}
	iter = param->mSubnetList->GetIterator();
	while(!iter->Done())
	{
		cEndpoint* t = (cEndpoint *)iter->GetData();
		mSubnetTree->Insert(t);
		iter->GetNext();
	}
	return true;
}

/*
 * cPbcastLayer::Deliver()
 *
 * Purpose:	Delivery interface function.
 * IN:		sender		-> The sender of the msg
 *			buffer		-> The buffer delivered up.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer delivers the message.
 * Return:	true if success, else false.
 */
bool cPbcastLayer::Deliver(cEndpoint* sender, cMsgBuffer* buffer, int messageType)
{
	cIterator*		iter;
	cDeliver*		deliver;
	int				size;

	// Remove and process header
	Header* header;
	buffer->GetPayload((void **)&header, &size);
	buffer->RemoveHeader(sizeof(Header));

	char* x = header->info.data.originator;

	switch(header->msgType)
	{

	case PBCAST_MSG_TYPE_FULL_GOSSIP_REQ:
		if(!sender->Equals(mMyNode->mEndpoint))
		{
			mGossipGroup->Clear();
			mGossipGroup->AddEndpoint(sender);
			_FullGossip();
		}
		// fall thru
	case PBCAST_MSG_TYPE_FULL_GOSSIP:
		if(!_HandleFullGossip(sender, header, buffer))
		{
			gError("Unable to handle gossip message.", __LINE__, __FILE__);
			return false;
		}
		break;

	case PBCAST_MSG_TYPE_MIN_GOSSIP:
		if(!_HandleMinGossip(sender, header, buffer))
		{
			gError("Unable to handle minimal gossip message.", __LINE__, __FILE__);
			return false;
		}
		break;

	case PBCAST_MSG_TYPE_DATA_DIRECT:
		mParam.mPbcastsRecvd++;
		size = sizeof(header->info.data.originator);
		sender = cEndpointFactory::AllocEndpoint(&x, &size);
		if(!sender)
		{
			return false;
		}
		if(!_HandleDataDissemination(sender, header, buffer))
		{
			gError("Unable to disseminate the message.", __LINE__, __FILE__);
		}
		sender->Release();
		break;

	case PBCAST_MSG_TYPE_DATA:
		mParam.mPbcastsRecvd++;
		size = sizeof(header->info.data.originator);
		sender = cEndpointFactory::AllocEndpoint(&x, &size);
		if(!sender)
		{
			return false;
		}
		if(!_HandleData(sender, header, buffer))
		{
			gError("Unable to handle data message.", __LINE__, __FILE__);
			return false;
		}
		sender->Release();
		break;

	case PBCAST_MSG_TYPE_RETR_REQ:
		if(!_HandleRetrReq(sender, header, buffer))
		{
			gError("Unable to handle retransmission request.", __LINE__, __FILE__);
			return false;
		}
		break;

	case PBCAST_MSG_TYPE_RETR:
		mParam.mNumRetransRecvd++;
		size = sizeof(header->info.data.originator);
		sender = cEndpointFactory::AllocEndpoint(&x, &size);
		if(!sender)
		{
			return false;
		}
		fflush(stdout);
		if(!_HandleData(sender, header, buffer))
		{
			gError("Unable to handle retransmitted message.", __LINE__, __FILE__);
			return false;
		}
		sender->Release();
		break;

	case PBCAST_MSG_TYPE_THRU:
		// Deliver to callbacks.
		mRegisterList->Lock();
		iter = mRegisterList->GetIterator();
		while(!iter->Done())
		{
			deliver = (cDeliver *)iter->GetData();
			deliver->Deliver(sender, buffer, messageType);
			iter->GetNext();
		}
		mRegisterList->Unlock();
		break;
	}
	return true;
}

/*
 * cPbcastLayer::ErrorCallback()
 *
 * Purpose:	Receives asynch error callbacks.
 * IN:		obj			-> Object param -- differs depending on type
 *			type		-> The type of this error callback.
 * OUT:		-
 * Cond:	-
 * PostCnd:	This layer either handles the error, or passes it up.
 * Return:	true if success, else false.
 */
bool cPbcastLayer::ErrorCallback(cObject* param, unsigned int type)
{
	cPbcastNode* resultNode;
	switch(type)
	{
		case ASYNCH_ERROR_EP_FAIL:
			gError("ASYNCH <pbcast>: Received endpoint fail notification for: ", __LINE__, __FILE__);

			// Make sure node is in our view.
			if(mView->Find((cEndpoint *)param, &resultNode))
			{
				mFailedNodes->Insert(resultNode);	// Add to failed nodes
				mView->Delete(resultNode);
				_ViewChange();
			}
			break;

		default:
			_DeliverErrorCallback(param, type);
			break;
	}
	return true;
}

/*
 * cPbcastLayer::_DeliverErrorCallback()
 *
 * Purpose:	Delivers asynch errors to any registrees.
 * IN:		obj		-> the object parameter
 *			type	-> the type parameter
 * OUT:		-
 * Return:	The local ip address.
 */
bool cPbcastLayer::_DeliverErrorCallback(cObject* obj, unsigned int type)
{
	cIterator*		iter;
	cErrorCallback* callback;

	iter = mErrorCallbackList->GetIterator();
	while(!iter->Done())
	{
		callback = (cErrorCallback *)iter->GetData();
		callback->ErrorCallback(obj, type);
		iter->GetNext();
	}
	return true;
}

/*
 * cPbcastLayer::RegisterViewCallback()
 *
 * Purpose:	Adds a view tracker.
 * IN:		handle		-> The pointer to the handle to use for registration.
 *			callback	-> The ViewHandler function.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The registration list increases in size.
 * Return:	true if success, else false.
 */
bool	cPbcastLayer::RegisterViewCallback(cHandle* handle, cView* callback)
{
	handle->mLayer = this;
	return mViewList->AddObject(handle, (cObject *)callback);
}

/*
 * cPbcastLayer::UnregisterViewCallback()
 *
 * Purpose:	Unregisters a view with this layer.
 * IN:		handle		-> The handle to find the registered
 * OUT:		-
 * Cond:	The handle specifies something in the registration list.
 * PostCnd:	The registration list decreases in size.
 * Return:	true if success, else false.
 */
bool	cPbcastLayer::UnregisterViewCallback(cHandle handle)
{
	if(handle.mLayer == this)
	{
		return mViewList->RemoveObject(handle);		
	}
	else if(mLayerBelow)
	{
		return mLayerBelow->UnregisterViewCallback(handle);
	}
	else
	{
		return false;
	}
}

/*
 * cPbcastLayer::_ProbTest()
 *
 * Purpose:	yes/no test based on given probability.
 * IN:		probVal		-> integer between 0 and RAND_MAX
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if success event should occur, else false.
 */
bool cPbcastLayer::_ProbTest(int probVal)
{
	return (rand() < probVal);
}

/*
 * cPbcastLayer::_ComputeProb()
 *
 * Purpose:	Computes the probVal for a given percentage.
 * IN:		percent		-> Percentage of time that event should happen.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	int, the probVal to use with ProbTest function.
 */
int	cPbcastLayer::_ComputeProb(double percent)
{
	double max = RAND_MAX;
	return (int) (percent * max);
}

/*
 * cPbcastLayer::_ViewChange()
 *
 * Purpose:	Publishes new view.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if success, else false.
 */
bool cPbcastLayer::_ViewChange()
{
	cView*					view;
	cViewIterator*			vIter;

	mView->UpdateDigest();	// Update the view digest on a view change.
	mStarted = true;

	mParam.mViewSize = mView->GetNumElements();
	mParam.mSubnetListSize = mSubnetTree->GetNumElements();

	cIterator* iter = mViewList->GetIterator();
	while(!iter->Done())
	{
		view = (cView *)iter->GetData();
		vIter = mView->GetViewIterator();
		view->ViewHandler(vIter);
		iter->GetNext();
	}

	// Compute new pbcast parameters
	_ComputeParams();
	_ResetRetransmitReqMsg();
	return true;
}